#!/bin/sh
# wg-system-reset.sh - Shell script to reset the system to factory default state.

BOOTCOUNT=1

_log_error() {
    logger -s -p local0.err -t "wg_system_reset" "ERROR: ${1}"
}

_log_notice() {
    logger -s -p local0.notice -t "wg_system_reset" "NOTICE: ${1}"
}

_cmp_fname() {
    if [ ! -z "${1}"  -a ! -z "${2}" ] ; then
        fname="`basename ${1}`"
        if [ "$fname" = "${2}" ] ; then
            return 0
        fi
    fi
    return 1
}

_is_system_file(){
    _cmp_fname "${1}" "system.json"
}

_is_interfaces_file(){
    _cmp_fname "${1}" "interfaces.json"
}

_is_ntpd_file(){
    _cmp_fname "${1}" "ntpd.json"
}

_is_sshd_file(){
    _cmp_fname "${1}" "sshd.json"
}

_on_exit() {
    _log_error "Reset to factory default failed. Rebooting..."
    rm -f "${WG_CONFIG_DIR}"/*.json
    rm -f "${WG_CONFIG_DIR}"/bootcount
    if [ ${BOOTCOUNT} -ge 3 ]; then
	    _log_error "Rebooted ${BOOTCOUNT} times: halting"
	    shutdown -h now
    fi
    BOOTCOUNT=$((${BOOTCOUNT} + 1))
    echo "BOOTCOUNT=${BOOTCOUNT}" > ${WG_CONFIG_DIR}/bootcount
    shutdown -r now
}

. /etc/default/wg_system

WG_PYTHON_BIN=${WG_BINDIR}/python
WG_CONFIG_PYC=${WG_BINDIR}/wg_system_config.pyc

WG_CONFIG_DIR=${WG_ETCDIR}/system
WG_DEFAULTS_DIR=${WG_CONFIG_DIR}/defaults
WG_TASKREQ_DIR=${WG_VARDIR}/system/task-requests
WG_TASKSTATUS_DIR=${WG_VARDIR}/system/task-status
WG_UPLOAD_DIR=${WG_VARDIR}/upload
WG_DATA_DIR=${WG_VARDIR}/data
WG_DB_DIR=${WG_VARDIR}/data/db
WG_TMP_DIR=${WG_VARDIR}/data/tmp

LOCALCERTSDIR=/usr/local/share/ca-certificates

[ -f ${WG_CONFIG_DIR}/bootcount ] && . ${WG_CONFIG_DIR}/bootcount

_log_notice "starting"

if [ ! -f "${WG_PYTHON_BIN}" -o ! -x "${WG_PYTHON_BIN}" ]; then
    _log_error "{WG_PYTHON_BIN} does not exist or is not an executable."
    exit 1
fi

# Set exit trap so we can restart wg_systemd if this script
# exits due to some failure.
trap _on_exit EXIT

_log_notice "Initializing configuration to default..."

# Cleanup configuration of all servers and postgres.
${WG_BINDIR}/wlogserver -uninstall -clean || true
${WG_BINDIR}/wlcollector -uninstall -clean || true
${WG_BINDIR}/wsserver -uninstall -clean || true
${WG_BINDIR}/wg_pg_service -uninstall -clean || true

# Clean up any pending task-requests and related status/uploaded data.
rm -f "${WG_TASKREQ_DIR}"/*.json
rm -f "${WG_TASKSTATUS_DIR}"/*.json
rm -f "${WG_UPLOAD_DIR}"/*

# Clean up default database directory
rm -rf "${WG_DB_DIR}"

# Clean up var tmp directory
rm -rf "${WG_TMP_DIR}"

# Clean up backup configuration directory
rm -f "${WG_ETCDIR}/backup"/*

# Clean up wgdevice_ssh configuration directory
rm -f "${WG_ETCDIR}/wgdevice_ssh"/*

# Clean up CA certificates installed locally
rm -f "${LOCALCERTSDIR}"/*.crt
if [ -x /usr/sbin/update-ca-certificates ] ; then
    /usr/sbin/update-ca-certificates || true
fi

# Clean up the wgdb configuration
rm -rf "${WG_ETCDIR}/wgdb"

# remove the local cherrypy sessions
rm -f "${WG_DATA_DIR}/session"/*

# Restore and apply the default system.json file.
_log_notice "Generating default ${WG_CONFIG_DIR}/system.json"
cp -p "${WG_DEFAULTS_DIR}"/system.json "${WG_CONFIG_DIR}"
"${WG_PYTHON_BIN}" "${WG_CONFIG_PYC}" "${WG_CONFIG_DIR}"/system.json

# Process the interfaces.json next
"${WG_PYTHON_BIN}" "${WG_CONFIG_PYC}" "${WG_CONFIG_DIR}"/interfaces.json

# Delete all remaining configuration files except
# system.json and interfaces.json file.
for f in "${WG_CONFIG_DIR}"/*.json ; do
    if _is_system_file "$f" || _is_interfaces_file "$f" ; then
        continue
    fi
    _log_notice "Deleting $f"
    rm -f "$f"
    "${WG_PYTHON_BIN}" "${WG_CONFIG_PYC}" "$f"
done

# Finally restore the remaining default configuration files.
for f in "${WG_DEFAULTS_DIR}"/*.json ; do
    if _is_system_file "$f" || _is_interfaces_file "$f" ; then
        continue
    fi
    fbase="`basename ${f}`"
    _log_notice "Generating default ${WG_CONFIG_DIR}/${fbase}"
    cp -p "$f" "${WG_CONFIG_DIR}"
    # For NTP service, just copy ntpd.json file and wg-ntpd.service will handle it later.
    # For SSHD service, just copy sshd.json file and wg-sshd.service will handle it later.
    if _is_ntpd_file "$f" || _is_sshd_file "$f" ; then
        continue
    fi
    "${WG_PYTHON_BIN}" "${WG_CONFIG_PYC}" "${WG_CONFIG_DIR}/${fbase}"
done

# Restore default database directory
if [ -d "${WG_DATA_DIR}" ] ; then
    mkdir "${WG_DB_DIR}"
    chown --reference="${WG_VARDIR}" "${WG_DB_DIR}"
    chmod --reference="${WG_VARDIR}" "${WG_DB_DIR}"
    # Restore the var tmp directory
    mkdir "${WG_TMP_DIR}"
    chown --reference="${WG_VARDIR}" "${WG_TMP_DIR}"
    chmod --reference="${WG_VARDIR}" "${WG_TMP_DIR}"
fi

# Next re-initialize all servers and postgres.
_log_notice "Initializing services..."
${WG_BINDIR}/wg_pg_service -install
sudo -u ${WG_ADMIN_USER} ${WG_BINDIR}/wlogserver -install
sudo -u ${WG_ADMIN_USER} ${WG_BINDIR}/wlcollector -install
sudo -u ${WG_ADMIN_USER} ${WG_BINDIR}/wsserver -install
# XXX:
# Method 1: If wgpr library is compiled with USE_SYSTEMD macro defined, use -configenable instead of -configstart to enable wsserver,
# that will not make Systemd start wsserver later but make Systemd start wsserver on next boot. So to start the wsserver in current boot,
# we should use dbus-send to call a method to wg-systemd-event-handler daemon which will start wsserver then.
#
# Method 2: If wgpr library is compiled without USE_SYSTEMD macro defined, just use existing -configstart option to start wsserver,
# that will generate wsserver SysV init script under /etc/init.d and the wsserver service will be also started on the first boot. 
#
# Currently the method 2 is our selection.
${WG_BINDIR}/wsserver -configstart

# Start the wg_backup upstart job
#initctl start -nq wg_backup || true
# XXX: For Systemd, always enable wg-backup.service

# Start the wgdevice_ssh upstart job
#initctl start -nq wgdevice_ssh || true
# XXX: For Systemd, always enable wgdevice-ssh-init.service and wgdevice-ssh.service

# Emit event to indicate system has been initialized.
#initctl emit --no-wait wg_event_system_ready
# XXX: For Systemd, just put services that depends on wg_event_system_ready after wg-system-init.service instead.

trap - EXIT

[ -f ${WG_CONFIG_DIR}/bootcount ] && rm -f ${WG_CONFIG_DIR}/bootcount
exit 0
