#!/bin/sh

_configure_ssh_add_user() {
    "$PYTHON" - "$@" <<EOF
import json, sys
user = sys.argv[1]
input = open(sys.argv[2], 'r')
output = open(sys.argv[3], 'w')
config = json.load(input)
if 'enabled' in config and config['enabled'] is True and not 'users' in config:
    config['users'] = ['$WG_ADMIN_USER']
if not 'users' in config:
    config['users'] = []
if not user in config['users']:
    config['users'].append(user)
config['enabled'] = True
string = json.dumps(config, sort_keys=True, indent=4)
output.write(string + '\n')
EOF
}

_configure_ssh_remove_user() {
    "$PYTHON" - "$@" <<EOF
import json, sys
user = sys.argv[1]
input = open(sys.argv[2], 'r')
output = open(sys.argv[3], 'w')
config = json.load(input)
if 'users' in config:
    if user in config['users']:
        config['users'].remove(user)
    if len(config['users']) == 0:
        config['enabled'] = False
string = json.dumps(config, sort_keys=True, indent=4)
output.write(string + '\n')
EOF
}

_configure_ssh_set_port() {
    "$PYTHON" - "$@" <<EOF
import json, sys
port = sys.argv[1]
input = open(sys.argv[2], 'r')
output = open(sys.argv[3], 'w')
config = json.load(input)
if not 'enabled' in config:
    config['enabled'] = False
config['port'] = int(port)
string = json.dumps(config, sort_keys=True, indent=4)
output.write(string + '\n')
EOF
}

_print_ssh_configuration() {
    "$PYTHON" - "$@" <<EOF
import json, sys
input = open(sys.argv[1])
config = json.load(input)
if not 'enabled' in config:
    config['enabled'] = False
if not 'port' in config:
    config['port'] = 22
if not 'users' in config:
    config['users'] = ['$WG_ADMIN_USER']
if config['enabled'] is True and '$WG_ADMIN_USER' in config['users']:
    print('Support Access enabled: true')
else:
    print('Support Access enabled: false')
print('Support Access port: %s' % config['port'])
EOF
}

_save_file() {
    /usr/bin/cmp -s "$1" "$2" || /bin/cp -f "$1" "$2"
}

configure_port() {
    if _configure_ssh_set_port "$1" "$SSH_FILE" "$TMP_SSH_FILE"; then
        _save_file "$TMP_SSH_FILE" "$SSH_FILE"
    fi
}

disable_support() {
    if _configure_ssh_remove_user "$WG_ADMIN_USER" "$SSH_FILE" "$TMP_SSH_FILE"; then
        _save_file "$TMP_SSH_FILE" "$SSH_FILE"
    fi
}

enable_support() {
    if _configure_ssh_add_user "$WG_ADMIN_USER" "$SSH_FILE" "$TMP_SSH_FILE"; then
        _save_file "$TMP_SSH_FILE" "$SSH_FILE"
    fi
}

parse_arguments() {
    ARGS=`getopt -o "dehp:" -l "disable,enable,help,port:" -n "$SCRIPT" -- "$@"`

    if [ $? -ne 0 ]; then
        print_usage_and_quit
    fi

    eval set -- "$ARGS"

    while true; do
        case "$1" in

            -d|--disable)
                ENABLED=false
                shift
                ;;

            -e|--enable)
                ENABLED=true
                shift
                ;;

            -h|--help)
                print_usage
                shift
                exit 0
                ;;

            -p|--port)
                PORT="$2"
                shift 2
                ;;

            --)
                shift
                break
                ;;

        esac
    done

    for ARG in "$@"; do
        print_warning unrecognized argument \'"$ARG"\'
    done
}

print_configuration() {
    _print_ssh_configuration "$SSH_FILE"
}

print_error_and_quit() {
    echo "$SCRIPT: $*" >&2
    exit 1
}

print_warning() {
    echo "$SCRIPT: $*" >&2
}

print_usage() {
    cat <<EOF >&2
Usage: $SCRIPT [OPTION]...

Enable or disable the Support Access service and/or configure the port
utilized, or display the configuration if none of -dehp is specified.

Options:

  -e,--enable       enable service
  -d,--disable      disable service
  -p,--port=PORT    configure service for port number PORT
  -h,--help         display this help and exit

Examples:

  $SCRIPT           # Display service configuration
  $SCRIPT -e        # Enable service
  $SCRIPT -d        # Disable service
  $SCRIPT -p 22     # Configure service for port 22
  $SCRIPT -e -p 22  # Enable service using port 22

Exit status:

 0  if OK,
 1  if error
EOF
}

print_usage_and_quit() {
    print_usage
    exit 1
}

set_configuration() {
    if [ -n "$ENABLED" ]; then
        if [ "$ENABLED" = true ]; then
            enable_support
        else
            disable_support
        fi
    fi
    if [ -n "$PORT" ]; then
        configure_port "$PORT"
    fi
}

validate_arguments() {
    if [ -n "$PORT" ]; then
        if ! echo $PORT | egrep '^[1-9][0-9]{0,4}$' >/dev/null; then
            print_error_and_quit "$PORT: Invalid protocol port"
        elif [ "$PORT" -lt 1 -o "$PORT" -gt 65535 ]; then
            print_error_and_quit "$PORT: Invalid protocol port"
        elif [ "$PORT" -eq 20 -o "$PORT" -eq 21 ]; then
            print_error_and_quit "$PORT: Reserved protocol port (FTP)"
        elif [ "$PORT" -eq 25 ]; then
            print_error_and_quit "$PORT: Reserved protocol port (SMTP)"
        elif [ "$PORT" -eq 53 ]; then
            print_error_and_quit "$PORT: Reserved protocol port (DNS)"
        elif [ "$PORT" -eq 68 ]; then
            print_error_and_quit "$PORT: Reserved protocol port (BOOTP)"
        elif [ "$PORT" -eq 80 -o "$PORT" -eq 443 ]; then
            print_error_and_quit "$PORT: Reserved protocol port (HTTP/HTTPS)"
        elif [ "$PORT" -eq 109 -o "$PORT" -eq 110 ]; then
            print_error_and_quit "$PORT: Reserved protocol port (POP2/POP3)"
        elif [ "$PORT" -eq 123 ]; then
            print_error_and_quit "$PORT: Reserved protocol port (NTP)"
        elif [ "$PORT" -eq 5432 ]; then
            print_error_and_quit "$PORT: Reserved protocol port (PostgreSQL)"
        fi
    fi
}

#########################
# Execution begins here #
#########################

set -eu
umask 002

SCRIPT="`basename $0`"

DEFAULTS=etc/default/wg_system

if [ -r /$DEFAULTS ]; then
    . /$DEFAULTS
elif [ -r $HOME/$DEFAULTS ]; then
    . $HOME/$DEFAULTS
fi

PYTHON="${WG_BINDIR:-}${WG_BINDIR:+/}python"
SSH_FILE="$WG_ETCDIR/system/sshd.json"

TMP_SSH_FILE=`mktemp`
trap "/bin/rm -f $TMP_SSH_FILE" EXIT QUIT

ENABLED=
PORT=

parse_arguments "$@"
validate_arguments
set_configuration
print_configuration

exit 0
