var IPHELPER = {};


/*
 *  IPv4 -------------------
 */


IPHELPER.validIP4 = function (ip4) {
    return (/^(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)$/i.test(ip4));
};

/*
 *  net4 : x.x.x.x/n
 */
IPHELPER.validNetwork4Syntax = function (net4) {
    var fields = net4.split('/');

    if (fields.length !== 2) {
        return false;
    }

    if (!IPHELPER.validIP4(fields[0])) {
        return false;
    }

    var val = parseInt(fields[1], 10);
    if (val < 0 || val > 32) {
        return false;
    }

    return true;
};

/*
 *  Same as valid_network4_syntax() but with more checking to see if it's
 *  actually an IPv4 network address.
 *
 *  net4 : x.x.x.x/n
 */
IPHELPER.validNetwork4 = function (net4) {
    var fields = net4.split('/');

    if (fields.length !== 2) {
        return false;
    }

    if (!IPHELPER.validIP4(fields[0])) {
        return false;
    }

    var val = parseInt(fields[1], 10);
    if (val < 0 || val > 32) {
        return false;
    }

    var net = IPHELPER.dqstrToInt(fields[0]);
    var mask = IPHELPER.dqstrToInt(IPHELPER.slashToNetmask(val));

    // bit mask turn to signed-32 bit, so use 0xFFFFFFFF to make the comparison work
    return (net & 0xFFFFFFFF) === (net & mask);
};

/*
 *  net : x.x.x.x1-x.x.x.x2
 */
IPHELPER.validRange4 = function (range4) {
    var fields = range4.split('-');

    if (fields.length !== 2) {
        return false;
    }

    if (!IPHELPER.validIP4(fields[0]) || !IPHELPER.validIP4(fields[1])) {
        return false;
    }

    if (IPHELPER.dqstrToInt(fields[0]) >= IPHELPER.dqstrToInt(fields[1])) {
        return false;
    }

    return true;
};

/*
 *  slash_addr4  :   x.x.x.x/n
 */
IPHELPER.validBroadcast4 = function (slash_addr4) {
    var fields = slash_addr4.split('/');

    if (fields.length !== 2) {
        return false;
    }

    if (!IPHELPER.validIP4(fields[0])) {
        return false;
    }

    var val = parseInt(fields[1], 10);
    if (val < 0 || val > 32) {
        return false;
    }

    var broadcast = IPHELPER.dqstrToInt(fields[0]);
    var mask = IPHELPER.dqstrToInt(IPHELPER.slashToNetmask(val));

    // bit mask turn to signed-32 bit, so use 0xFFFFFFFF to make the comparison work
    return (~mask | (broadcast & mask)) === (broadcast & 0xFFFFFFFF);
};

/*
 *  Return an integer value given an IP address as dotted-quad string
 *  - "2.2.2.2" -> a corresponding integer
 *  - return null if it's not valid
 */
IPHELPER.dqstrToInt = function (dq) {
    var octets = dq.split('.');

    if (octets.length !== 4) {
        return null;
    }

    var vals = [];
    var i;
    for (i = 0; i < 4; i++) {
        vals[i] = parseInt(octets[i], 10);
        if (vals[i] > 255) {
            return null;
        }
    }

    return ((((((+octets[0]) * 256) + (+octets[1])) * 256) + (+octets[2])) * 256) + (+octets[3]);
};

/*
 *  Return a dotted-quad string given an integer.
 *  An integer -> "2.2.2.2"
 */
IPHELPER.intToDqstr = function (intval) {
    var vals = [];
    vals[0] = (intval >> 24) & 0x000000ff;
    vals[1] = (intval & 0x00ff0000) >> 16;
    vals[2] = (intval & 0x0000ff00) >> 8;
    vals[3] = (intval & 0x000000ff);

    return vals[0] + '.' +  vals[1] + '.' + vals[2] + '.' + vals[3];
};

/*
 *  Convert a netmask address (x.x.x.x) to a corresponding slash (n).
 */
IPHELPER.netmaskToSlash = function (dqstr) {

    if (!IPHELPER.validIP4(dqstr)) {
        return null;
    }

    var n = IPHELPER.dqstrToInt(dqstr);
    var i = 0;
    while ((i < 32) && (n % 2 === 0)) {
        n = n >> 1;
        i += 1;
    }

    // Subtract the trailing zeros from 32
    return 32 - i;
};

/*
 *  Convert slash (n) to a corresponding netmask address, x.x.x.x.
 *  return null if it is invalid.
 */
IPHELPER.slashToNetmask = function (nslash) {
    // make sure the slash is 0-32
    if (nslash < 0 || nslash > 32) {
        return null;
    }

    var nmask = 0;

    // Convert the slash to a netmask.  It constructs a 32-bit int with the
    // first nSlash bits set to 1.  The result is the corresponding netmask.
    if (nslash === 0) {
        nmask = 0;
    } else {
        nmask = (0xffffffff  << (32 - nslash));
    }

    return IPHELPER.intToDqstr(nmask);
};

/*
 *  Return x.x.x.x/n
 */
IPHELPER.toSlashAddr4 = function (ip_dqstr, mask_dqstr) {
    // remove leading, inner, and trailing spaces
    ip_dqstr = ip_dqstr.replace(/\s+/g, '');
    mask_dqstr = mask_dqstr.replace(/\s+/g, '');

    if (!IPHELPER.validIP4(ip_dqstr) || !IPHELPER.validIP4(mask_dqstr)) {
        return null;
    }

    return ip_dqstr + '/' + IPHELPER.netmaskToSlash(mask_dqstr);
};

IPHELPER.sizeOfNetwork4 = function (net4) {
    if (!IPHELPER.validNetwork4Syntax(net4)) {
        return -1;
    }

    var fields = net4.split('/');
    var val = parseInt(fields[1], 10);

    return 1 << (32 - val);
};

IPHELPER.sizeOfRange4 = function (range4) {
    if (!IPHELPER.validRange4(range4)) {
        return -1;
    }

    var fields = range4.split('-');
    var val1 = IPHELPER.dqstrToInt(fields[0]);
    var val2 = IPHELPER.dqstrToInt(fields[1]);

    return val2 - val1 + 1;
};

IPHELPER.isRange4Overlap = function (ipA1, ipA2, ipB1, ipB2) {
    var decA1 = IPHELPER.dqstrToInt(ipA1);
    var decA2 = IPHELPER.dqstrToInt(ipA2);
    var decB1 = IPHELPER.dqstrToInt(ipB1);
    var decB2 = IPHELPER.dqstrToInt(ipB2);

    return (decA1 >= decB1 && decA1 <= decB2) ||
           (decA2 >= decB1 && decA2 <= decB2) ||
           (decB1 >= decA1 && decB1 <= decA2) ||
           (decB2 >= decA1 && decB2 <= decA2);

};

/* Determines if 2 networks are sharing the same subnet, by default will check the first 3 octects (24 bits) */
IPHELPER.isRange4OnSameSubnet = function (ip1, ip2, numBits) {
    if (!numBits) {
        numBits = 24;
    }

    var numCommonBits = IPHELPER.getCommonBitsMask4(ip1, ip2);
    return numCommonBits >= numBits;
};

/*  192.168.53.211 ~ 192.168.53.1 -> 24  */
IPHELPER.getCommonBitsMask4 = function (ip1, ip2) {
    var bitStr1 = IPHELPER.ip4ToBitStr(ip1);
    var bitStr2 = IPHELPER.ip4ToBitStr(ip2);
    var i;
    for (i = 0; i < 32; i++) {
        if (bitStr1.charAt(i) !== bitStr2.charAt(i)) {
            return i;
        }
    }

    return 32;
};

/* 255.255.255.0 -> 11111111 11111111 11111111 00000000  */
IPHELPER.ip4ToBitStr = function (ipAddress) {
    var bitStr = '';
    var parts = ipAddress.split('.');
    var i,
        val,
        str;
    for (i = 0; i < parts.length; i++) {
        val = parseInt(parts[i], 10);
        str = val.toString(2);
        while (str.length < 8) { // fill in the leading 0's'
            str = '0' + str;
        }

        bitStr += str;
    }

    return bitStr;
};

/*
 *  hostAddr  x.x.x.x
 *
 *  return 0 if it's OK
 *         1 invalid IPv4 address
 *         2 if it's 0.0.0.0
 *         3 if it's 255.255.255.255
 *         4 the first octet is 0 or 127, reserved address space
 *         5 multicast address (the first octet is > 223 and < 240)
 *         6 reserved for future use (range 240.0.0.0 through 255.255.255.254)
 *
 *         Error msgs
 *         invalidHostAddress   -- {0} is an invalid host IPv4 address.
 *         zeroHostAddress      -- 0.0.0.0 is not allowed.
 *         twoFiveFiveHostAddress -- 255.255.255.255 is not allowed.
 *         reservedAddressSpace -- {0} is not a host IPv4 because its first octet either equals to 0 or 127 which are reserved address spaces.
 *         multicastAddressSpace -- {0} is not a host IP because its first octet is between 224 and 239 (inclusive) which are reserved address spaces for multicast.
 */
IPHELPER.usableHostIP4 = function (hostAddr, allowMulticast, allowReservedForFuture) {
    allowMulticast = allowMulticast || false;
    allowReservedForFuture = allowReservedForFuture || false;

    hostAddr = $.trim(hostAddr);

    if (!IPHELPER.validIP4(hostAddr)) {
        return 1;
    }

    if (hostAddr === '0.0.0.0') {
        return 2;
    }
    if (hostAddr === '255.255.255.255') {
        return 3;
    }

    var fields = hostAddr.split('.');
    var firstOctet = parseInt(fields[0], 10);
    if ((firstOctet === 0) || (firstOctet === 127)) {
        return 4;
    }

    if (!allowMulticast) {
        if (firstOctet > 223 && firstOctet < 240) {
            return 5;
        }
    }

    if (!allowReservedForFuture) {
        var ipInt = IPHELPER.dqstrToInt(hostAddr);
        if ((ipInt >= IPHELPER.dqstrToInt("240.0.0.0")) &&
                (ipInt <= IPHELPER.dqstrToInt("255.255.255.254"))) {
            return 6;
        }
    }

    return 0;
};

/*
 *  hostAddr  x.x.x.x/n
 *
 *  return 0 if it's OK
 *         1 invalid IPv4 address
 *         2 if it's 0.0.0.0
 *         3 if it's 255.255.255.255
 *         4 the first octet is 0 or 127, reserved address space
 *         5 multicast address (the first octet is > 223 and < 240)
 *         6 reserved for future use (range 240.0.0.0 through 255.255.255.254)
 *         7 cannot be a network IP
 *         8 cannot be a broadcast IP
 *         9 cannot the netmask is 0
 *         Error msgs
 *         invalidHostAddress   -- {0} is an invalid host IPv4 address.
 *         zeroHostAddress      -- 0.0.0.0 is not allowed.
 *         twoFiveFiveHostAddress -- 255.255.255.255 is not allowed.
 *         reservedAddressSpace -- {0} is not a host IPv4 because its first octet either equals to 0 or 127 which are reserved address spaces.
 *         multicastAddressSpace -- {0} is not a host IP because its first octet is between 224 and 239 (inclusive) which are reserved address spaces for multicast.
 *         isNetworkIP -- {0} is a network IP.
 *         isBroadcastIP -- "{0} is a broadcast IP.
 *         invalidHostNetMaskZero -- {0} is not a host IP because its netmask is 0.
 */
IPHELPER.usableHostIP4WithSlash = function (hostAddrWithSlash, allowMulticast, allowReservedForFuture) {
    allowMulticast = allowMulticast || false;
    allowReservedForFuture = allowReservedForFuture || false;

    hostAddrWithSlash = $.trim(hostAddrWithSlash);

    var slash = hostAddrWithSlash.split('/');
    if (slash.length !== 2) {
        return 1;
    }

    if (IPHELPER.validNetwork4(hostAddrWithSlash)) {
        return 7;
    }

    var ip = slash[0];
    var nmask = parseInt(slash[1], 10);

    if (nmask === 0) {
        return 9;
    }

    var nBroadcast = IPHELPER.dqstrToInt(ip);
    var nMask = IPHELPER.dqstrToInt(IPHELPER.slashToNetmask(nmask));

    if ((~nMask | (nBroadcast & nMask)) === (nBroadcast & 0xFFFFFFFF)) { //@todo check this
        return 8;
    }

    // continue with the host check
    return IPHELPER.usableHostIP4(ip, allowMulticast, allowReservedForFuture);
};

/*
 *   net4 : x.x.x.x/n
 *   return 0: OK
 *          1 invalid Network IPv4 address
 *          2 if it's 0.0.0.0
 *          3 if it's 255.255.255.255
 *          4 the first octet is 0 or 127, reserved address space
 *          5 multicast address (the first octet is > 223 and < 240)
 *          6 reserved for future use (range 240.0.0.0 through 255.255.255.254)
 */
IPHELPER.usableNetworkIP4 = function (net4, allowMulticast, allowReservedForFuture) {
    allowMulticast = allowMulticast || false;
    allowReservedForFuture = allowReservedForFuture || false;

    net4 = $.trim(net4);
    if (!IPHELPER.validNetwork4(net4)) {
        return 1;
    }

    var fields = net4.split('/');
    var ip = fields[0];

    if (ip === '0.0.0.0') {
        return 2;
    }
    if (ip === '255.255.255.255') {
        return 3;
    }

    var octs = ip.split('.');
    var firstOctet = parseInt(octs[0], 10);
    if ((firstOctet === 0) || (firstOctet === 127)) {
        return 4;
    }

    if (!allowMulticast) {
        if (firstOctet > 223 && firstOctet < 240) {
            return 5;
        }
    }

    if (!allowReservedForFuture) {
        var ipInt = IPHELPER.dqstrToInt(ip);
        if ((ipInt >= IPHELPER.dqstrToInt("240.0.0.0")) &&
                (ipInt <= IPHELPER.dqstrToInt("255.255.255.254"))) {
            return 6;
        }
    }

    return 0;
};

/*
 * rangeAddr: x.x.x.x - x.x.x.y
 *
 * return 0: OK
 *          1 not a valid range IPv4 address
 *          2 if it's 0.0.0.0
 *          3 if it's 255.255.255.255
 *          4 the first octet is 0 or 127, reserved address space
 *          5 multicast address (the first octet is > 223 and < 240)
 *          6 reserved for future use (range 240.0.0.0 through 255.255.255.254)
 */
IPHELPER.usableRangeIP4 = function (range4, allowMulticast, allowReservedForFuture) {
    allowMulticast = allowMulticast || false;
    allowReservedForFuture = allowReservedForFuture || false;

    range4 = $.trim(range4);
    if (!IPHELPER.validRange4(range4)) {
        return 1;
    }

    var fields = range4.split('-');
    var ip1 = fields[0];
    var ip2 = fields[1];

    var ret = IPHELPER.usableHostIP4(ip1, allowMulticast, allowReservedForFuture);
    if (ret !== 0) {
        return ret;
    }

    ret = IPHELPER.usableHostIP4(ip2, allowMulticast, allowReservedForFuture);
    if (ret !== 0) {
        return ret;
    }

    return 0;
};

/*
 *  trim leading and trialing spaces.
 *  remove inner spaces
 *  needs to be in x.x.x.x or x.x.x.x/y format, if not, return
 *  remove leading 0 except when it's 0
 *
 *  1.1.1.1 ==> 1.1.1.1
 *  01.01.01.01 => 1.1.1.1
 *  1.0.0.1 ==> 1.0.0.1
 *  1. 01. 0 1 .01 => 1.1.1.1
 *  1.1.00.0/024 ==> 1.1.0.0/24
 *
 * ip4: string
 */
IPHELPER.trimIP4 = function (ip4) {
    ip4 = $.trim(ip4);
    ip4 = ip4.split(' ').join(''); // remove inner spaces

    var fields = ip4.split('.');
    if (fields.length !== 4) {
        return ip4;
    }

    var ipv4_addr = ip4.split('/');
    if (!IPHELPER.validIP4(ipv4_addr[0])) {
        return ip4;
    }

    var slash = fields[3].split('/');
    var mask = null;
    if (slash.length === 2) {
        fields[3] = slash[0];
        mask = slash[1];
    }

    ip4 = IPHELPER.trimLeadingZero(fields[0]) + '.' +
          IPHELPER.trimLeadingZero(fields[1]) + '.' +
          IPHELPER.trimLeadingZero(fields[2]) + '.' +
          IPHELPER.trimLeadingZero(fields[3]);

    if (mask) {
        mask = IPHELPER.trimLeadingZero(mask);
        ip4 += '/' + mask;
    }

    return ip4;
};

// if it's 0 do not trim it
// s: string
IPHELPER.trimLeadingZero = function (s) {
    while (s.substr(0, 1) === '0' && s.length > 1) {
        s = s.substr(1);
    }

    return s;
};

/*
 *  IPv6 -------------------
 */

/*
 *  Return true for an IPv6 full/abbreviated address (x:x:x:x:x:x:x:x), e.g.,
 *  fe80:0000:0000:0000:0202:b3ff:fe1e:8329, fe80::202:b3ff:fe1e:8329.
 *
 *  Return false for an invalid IPv6 address string and an IPv6 with slash
 *
 *  Notes: There is no checking against reserved addresses.
 */
IPHELPER.validIP6 = function (ip6) {
    return (/^((([0-9A-Fa-f]{1,4}:){7}([0-9A-Fa-f]{1,4}|:))|(([0-9A-Fa-f]{1,4}:){6}(:[0-9A-Fa-f]{1,4}|:))|(([0-9A-Fa-f]{1,4}:){5}(((:[0-9A-Fa-f]{1,4}){1,2})|:))|(([0-9A-Fa-f]{1,4}:){4}(((:[0-9A-Fa-f]{1,4}){1,3})|:))|(([0-9A-Fa-f]{1,4}:){3}(((:[0-9A-Fa-f]{1,4}){1,4})|:))|(([0-9A-Fa-f]{1,4}:){2}(((:[0-9A-Fa-f]{1,4}){1,5})|:))|(([0-9A-Fa-f]{1,4}:){1}(((:[0-9A-Fa-f]{1,4}){1,6})|:))|(:(((:[0-9A-Fa-f]{1,4}){1,7})|:)))$/.test(ip6));
};

/*
 *  net6 : x.x.x.x/n
 */
IPHELPER.validNetwork6Syntax = function (net6) {
    var fields = net6.split('/');

    if (fields.length !== 2) {
        return false;
    }

    if (!IPHELPER.validIP6(fields[0])) {
        return false;
    }

    var val = parseInt(fields[1], 10);
    if (val < 1 || val > 128) {
        return false;
    }

    return true;
};

/*
 *  net6: x::0/n
 */
IPHELPER.validNetwork6 = function (net6) {
    var splits = net6.split('/');

    if (splits.length !== 2) {
        return false;
    }

    var prefix = parseInt(splits[1], 10);
    if (prefix === 128) {// an address with 128 prefix is considered a host address
        return false;
    }

    var bytes = IPHELPER.ipv6StrToBytes(splits[0]);
    if (bytes === null || bytes.length !== 16) {
        return false;
    }

    var prefixes = IPHELPER.ipv6PrefixToByteArray(prefix);
    if (prefixes === null || prefixes.length !== 16) {
        return false;
    }

    var results = [];
    var i;
    for (i = 0; i < 16; i++) {
        results[i] = bytes[i] & prefixes[i];
    }

    // compare
    for (i = 0; i < 16; i++) {
        if (bytes[i] !== results[i]) {
            return false;
        }
    }

    return true;
};

/*
 *  range6: x::x-x::y
 */
IPHELPER.validRange6 = function (range6) {
    var fields = range6.split('-');

    if (fields.length !== 2) {
        return false;
    }

    if (!IPHELPER.validIP6(fields[0]) || !IPHELPER.validIP6(fields[1])) {
        return false;
    }

    if (IPHELPER.compareIPv6Addrs(fields[0], fields[1]) >= 0) {
        return false;
    }

    return true;
};

IPHELPER.getFullHexString = function (groupStr) {
    if (groupStr === null) {
        groupStr = "";
    }

    var hexString = "";
    var len = 4 - groupStr.length;
    while (len-- > 0) {
        hexString += "0";
    }

    return hexString + groupStr;
};

IPHELPER.getIPv4AddressFrom17To48Bit = function (ipv6Groups) {
    if (ipv6Groups === null || ipv6Groups.length !== 8) {
        return "";
    }

    var ipv4_addr = "";

    var str1 = IPHELPER.getFullHexString(ipv6Groups[1]);
    ipv4_addr += parseInt(str1.substr(0, 2), 16);
    ipv4_addr += ".";
    ipv4_addr += parseInt(str1.substring(2), 16);
    ipv4_addr += ".";

    var str2 = IPHELPER.getFullHexString(ipv6Groups[2]);
    ipv4_addr += parseInt(str2.substr(0, 2), 16);
    ipv4_addr += ".";
    ipv4_addr += parseInt(str2.substring(2), 16);

    return ipv4_addr;
};

/*
 *  return null if the ipv6Addr is not an valid IPv6 address
 */
IPHELPER.ipv6AddrStrTo8Groups = function (ipv6AddrStr) {
    if (ipv6AddrStr === null) {
        return null;
    }

    // get rid of spaces
    ipv6AddrStr = ipv6AddrStr.replace(/\s+/g, '');

    // cannot contain more than 2 of ::12::
    var count = ipv6AddrStr.match(/::/g);
    if (count && count.length >= 2) {
        return null;
    }

    // cannot contain 1 or more of :::
    count = ipv6AddrStr.match(/:::/g);
    if (count && count.length >= 1) {
        return null;
    }

    var tokens = [];
    var begIndx = -1;
    var i;
    for (i = 0; i < ipv6AddrStr.length; i++) {
        if (i === ipv6AddrStr.length - 1) { // the end
            if (begIndx !== -1) {
                tokens.push(ipv6AddrStr.substring(begIndx, i + 1));
            } else {
                tokens.push(ipv6AddrStr.charAt(i));
            }
        } else {
            if (ipv6AddrStr.charAt(i) === ':') {
                if (begIndx !== -1) {
                    tokens.push(ipv6AddrStr.substring(begIndx, i));
                }

                tokens.push(':');
                begIndx = -1;
            } else {
                if (begIndx === -1) {
                    begIndx = i;
                }
            }
        }
    }

    if (tokens.length > 15) {
        return null;
    }

    var hexStrs = [];
    var doubleColonIdx = -1;
    var token = '';
    var prevToken = '';
    for (i = 0; i < tokens.length; i++) {
        prevToken = token;
        token = tokens[i];
        if (token === ':') {
            if (prevToken === ':') {
                doubleColonIdx = hexStrs.length;
            } else if (prevToken !== '') {
                hexStrs.push(prevToken);
            }
        }
    }

    if (prevToken === ':') {
        if (token === ':') {
            doubleColonIdx = hexStrs.length;
        } else {
            hexStrs.push(token);
        }
    }

    if (doubleColonIdx !== -1) {
        var hexStrsLength = 8;
        var newHexStrs = [];
        for (i = 0; i < hexStrsLength; i++) {
            newHexStrs[i] = null;
        }

        var inserts = hexStrsLength - hexStrs.length;
        for (i = 0; i < inserts; i++) {
            newHexStrs[doubleColonIdx + i] = '0';
        }
        var j;
        for (i = 0; i < hexStrs.length; i++) {
            for (j = 0; j < newHexStrs.length; j++) {
                if (newHexStrs[j] === null) {
                    newHexStrs[j] = hexStrs[i];
                    break;
                }
            }
        }

        hexStrs = newHexStrs;
    }

    if (hexStrs.length !== 8) {
        return null;
    }

    // check if each cannot contain more than 4
    for (i = 0; i < hexStrs.length; i++) {
        if (hexStrs[i].length > 4) {
            return null;
        }

        if (!IPHELPER.isHexStr(hexStrs[i])) {
            return null;
        }
    }

    return hexStrs;
};

/*
 *  return list of Number. The list size is 16. Each number represents a byte.
 */
IPHELPER.ipv6StrToBytes = function (ipv6AddrStr) {
    var hexStrs = IPHELPER.ipv6AddrStrTo8Groups(ipv6AddrStr);
    if (hexStrs === null) {
        return null;
    }

    var ret = [];
    var i;
    for (i = 0; i < 16; i++) {
        ret[i] = 0;
    }
    var hexWord,
        hexWordLength,
        hexWordIndex,
        firstByte,
        secondByte,
        charValue;
    for (i = 0; i < hexStrs.length; i++) {
        hexWord = hexStrs[i];
        hexWordLength = hexWord.length;
        hexWordIndex = 0;

        firstByte = 0;
        secondByte = 0;

        // high order 4 bits of first byte
        if (hexWordLength > 3) {
            charValue = parseInt(hexWord[hexWordIndex], 16);
            hexWordIndex += 1;
            firstByte = firstByte | charValue << 4;
        }

        // low order 4 bits of the first byte
        if (hexWordLength > 2) {
            charValue = parseInt(hexWord[hexWordIndex], 16);
            hexWordIndex += 1;
            firstByte = firstByte | charValue;
        }

        // high order 4 bits of second byte
        if (hexWordLength > 1) {
            charValue = parseInt(hexWord[hexWordIndex], 16);
            hexWordIndex += 1;
            secondByte = secondByte | charValue << 4;
        }

        // low order 4 bits of the first byte """
        charValue = parseInt(hexWord[hexWordIndex], 16);
        hexWordIndex += 1;
        secondByte = secondByte | charValue;

        ret[i * 2] = firstByte;
        ret[i * 2 + 1] = secondByte;
    }

    return ret;
};

/*
 *
 */
IPHELPER.bytesToIPv6Str = function (ipByteArray) {
    var _hex_chars = ['0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
        'a', 'b', 'c', 'd', 'e', 'f'];

    if (ipByteArray.length !== 16) {
        return null;
    }

    var bufferStr = '';
    var hasLeadingZero = true;
    var j, i;
    var _char;
    for (i = 0; i < ipByteArray.length; i++) {
        if (i % 2 === 0) {
            hasLeadingZero = true;
        }

        j = (ipByteArray[i] & 0xf0) >> 4;
        _char = _hex_chars[j];
        if (_char === '0') {
            if (!hasLeadingZero) {
                bufferStr = bufferStr + _char;
            }
        } else {
            hasLeadingZero = false;
            bufferStr = bufferStr + _char;
        }

        j = ipByteArray[i] & 0x0f;
        _char = _hex_chars[j];
        if (_char === '0') {
            if (!hasLeadingZero) {
                bufferStr = bufferStr + _char;
            } else if (i % 2 !== 0) {
                bufferStr = bufferStr + _char; // last zero
            }
        } else {
            hasLeadingZero = false;
            bufferStr = bufferStr + _char;
        }

        if (i % 2 !== 0 && ((i + 1) < ipByteArray.length)) {
            bufferStr = bufferStr + ':';
        }
    }

    // condensing the notation
    var beg = -1;
    var end = -1;
    var consecutiveZeroes = false;
    var splits = bufferStr.split(':');
    for (i = splits.length - 1; i >= 0; i--) { //decreasing index
        if (splits[i] === '0') {
            if (end === -1) {
                end = i;
                beg = i;
            } else {
                if (!consecutiveZeroes) {
                    beg = i;
                }
            }
        } else {
            if (end !== -1) {
                if (beg === end) { // do not condense single zero
                    end = -1;
                    beg = -1;
                } else {
                    consecutiveZeroes = true;
                }
            }
        }
    }

    if (beg === end) { // do not condense single zero
        end = -1;
        beg = -1;
    }

    // write out the condensed notation
    bufferStr = '';
    var prev = '';
    for (i = 0; i < splits.length; i++) {
        if (i >= beg && i < end) {
            continue;
        }
        if (i === end) {
            if (prev !== ':') {
                bufferStr = bufferStr + ':';
            }

            bufferStr = bufferStr + ':';
        } else {
            bufferStr = bufferStr + splits[i];
            prev = splits[i];

            if (i !== (splits.length - 1)) {
                bufferStr = bufferStr + ':';
                prev = ':';
            }
        }
    }

    return bufferStr;
};

/*
 *
 */
IPHELPER.isSameIPv6Addr = function (ipv6Addr1, ipv6Addr2) {
    var prefix1 = 128;
    var fields1 = ipv6Addr1.split('/');
    var i;

    if (fields1.length === 2) {
        ipv6Addr1 = fields1[0];
        prefix1 = parseInt(fields1[1], 10);
    }

    var prefix2 = 128;
    var fields2 = ipv6Addr2.split('/');
    if (fields2.length === 2) {
        ipv6Addr2 = fields2[0];
        prefix2 = parseInt(fields2[1], 10);
    }

    var bytes1 = IPHELPER.ipv6StrToBytes(ipv6Addr1);
    var bytes2 = IPHELPER.ipv6StrToBytes(ipv6Addr2);

    if (bytes1 === null || bytes1.length !== 16) {
        return false;
    }

    if (bytes2 === null || bytes2.length !== 16) {
        return false;
    }

    for (i = 0; i < bytes1.length; i++) {
        if (bytes1[i] !== bytes2[i]) {
            return false;
        }
    }

    if (prefix1 !== prefix2) {
        return false;
    }

    return true;
};

/*
 *  return -1, if ipv6Addr1 is lowever
 *          0, if both are the same
 *          1, if ipv6Addr1 is higher
 */
IPHELPER.compareIPv6Addrs = function (ipv6Addr1, ipv6Addr2) {
    var prefix1 = 128;
    var fields1 = ipv6Addr1.split('/');
    var i;
    if (fields1.length === 2) {
        ipv6Addr1 = fields1[0];
        prefix1 = parseInt(fields1[1], 10);
    }

    var prefix2 = 128;
    var fields2 = ipv6Addr2.split('/');
    if (fields2.length === 2) {
        ipv6Addr2 = fields2[0];
        prefix2 = parseInt(fields2[1], 10);
    }

    var bytes1 = IPHELPER.ipv6StrToBytes(ipv6Addr1);
    var bytes2 = IPHELPER.ipv6StrToBytes(ipv6Addr2);

    if (bytes1 === null || bytes1.length !== 16) {
        return -1;
    }

    if (bytes2 === null || bytes2.length !== 16) {
        return 1;
    }

    for (i = 0; i < bytes1.length; i++) {
        if (bytes1[i] < bytes2[i]) {
            return -1;
        }
        if (bytes1[i] > bytes2[i]) {
            return 1;
        }
    }

    if (prefix1 < prefix2) {
        return -1;
    }
    if (prefix1 > prefix2) {
        return 1;
    }

    return 0;
};

/*
 *
 */
IPHELPER.isIPv6RangeOverlap = function (ipv6A1, ipv6A2, ipv6B1, ipv6B2) {
    return (((IPHELPER.compareIPv6Addrs(ipv6A1, ipv6B1) >= 0) && (IPHELPER.compareIPv6Addrs(ipv6A1, ipv6B2) <= 0)) ||
             ((IPHELPER.compareIPv6Addrs(ipv6A2, ipv6B1) >= 0) && (IPHELPER.compareIPv6Addrs(ipv6A2, ipv6B2) <= 0)) ||
             ((IPHELPER.compareIPv6Addrs(ipv6B1, ipv6A1) >= 0) && (IPHELPER.compareIPv6Addrs(ipv6B1, ipv6A2) <= 0)) ||
             ((IPHELPER.compareIPv6Addrs(ipv6B2, ipv6A1) >= 0) && (IPHELPER.compareIPv6Addrs(ipv6B2, ipv6A2) <= 0)));
};

/*
 *  list size 16 of Number where each number represents a byte.
 */
IPHELPER.ipv6PrefixToByteArray = function (prefix) {
    if (prefix < 1 || prefix > 128) {
        return null;
    }

    // initialize the list of size 16 to 0
    var ret = [];
    var i;
    for (i = 0; i < 16; i++) {
        ret[i] = 0;
    }

    var byteNo = prefix / 8;
    for (i = 0; i < ret.length; i++) {
        if (i < byteNo) {
            ret[i] = 0xff;
        } else if (i === byteNo) {
            ret[i] = 0xff & (0xff << (8 - (prefix % 8)));
        }
    }

    return ret;
};

/*
 *
 */
IPHELPER.ipv6NetAddrFromSlashAddr = function (ipv6AddrWithSlash) {
    var splits = ipv6AddrWithSlash.split('/');
    if (splits.length !== 2) {
        return null;
    }

    return IPHELPER.ipv6NetAddr(splits[0], parseInt(splits[1], 10));
};

/*
 *
 */
IPHELPER.ipv6NetAddr = function (ipv6Addr, prefix) {
    if (prefix < 1 || prefix > 128) {
        return null;
    }

    var bytes = IPHELPER.ipv6StrToBytes(ipv6Addr);
    if (bytes === null || bytes.length !== 16) {
        return null;
    }

    var prefixes = IPHELPER.ipv6PrefixToByteArray(prefix);
    if (prefixes === null || prefixes.length !== 16) {
        return null;
    }

    var results = [];
    var i;
    for (i = 0; i < 16; i++) {
        results[i] = 0;
    }

    for (i = 0; i < results.length; i++) {
        results[i] = bytes[i] & prefixes[i];
    }

    return IPHELPER.bytesToIPv6Str(results);
};

/*
 *
 */
IPHELPER.isHexStr = function (str) {
    return (str.length && !isNaN(parseInt(str, 16)));
};


// jqgrid sort function for sorting columns with IPv4 or IPv6 data
// returns string representation of ip address
//   ipv4 addresses are front-zero-padded to 11 chars (always start with 0)
//   ipv6 addresses are fully expanded hex
IPHELPER.sortIpAddressCell = function (cell, rowObject) {
    var orgVal = cell;

    // remove the slash, if there is one
    var slash = cell.indexOf('/');
    if (slash > 0) {
        cell = cell.substring(0, slash);
    }
    var i;
    var empty_str = '';
    // IPv4 case
    var intVal = IPHELPER.dqstrToInt(cell);
    if (intVal) {
        // pad to 11 char string
        intVal = intVal + empty_str;
        var pad = 11 - intVal.length;
        for (i = 0; i < pad; i++) {
            intVal = '0' + intVal;
        }
        return intVal;
    }

    // IPv6 case
    var hexArray = IPHELPER.ipv6AddrStrTo8Groups(cell);
    if (hexArray) {
        var ret = '';

        for (i = 0; i < hexArray.length; i++) {
            ret += hexArray[i];
        }
        return ret;
    }

    // not IPv4 or IPv6, return original value
    return orgVal;
};
