-- This file contains the SQL statements necessary to create the 'base'
-- tables in the public schema for Dimension. It is intended to be run
-- by the log server (or other) when it initially starts operation.

SET SEARCH_PATH TO public;
CREATE SEQUENCE appliance_id_seq start with 1;

CREATE TABLE g_deviceinfo(
   id               INTEGER DEFAULT NEXTVAL('appliance_id_seq') PRIMARY KEY,
   ip_address       TEXT NOT NULL,
   appliance_name   TEXT NOT NULL,
   type             SMALLINT NOT NULL,
   sn               TEXT NOT NULL,
   model_name       TEXT,
   timezone         TEXT,
   version          TEXT,
   active           SMALLINT DEFAULT 0,
   cluster          TEXT,
   when_join        TIMESTAMP,
   update_time      TIMESTAMP
) WITH (FILLFACTOR = 50);
CREATE INDEX g_deviceinfo_sn_idx ON g_deviceinfo(sn);
CREATE INDEX g_deviceinfo_cl_idx ON g_deviceinfo("cluster");

CREATE TABLE g_clusterinfo(
  id                INTEGER DEFAULT NEXTVAL('appliance_id_seq') PRIMARY KEY,
  cluster_name      TEXT NOT NULL,
  update_time       TIMESTAMP
) WITH (FILLFACTOR = 50);
CREATE UNIQUE INDEX g_clusterinfo_cn_idx ON g_clusterinfo(cluster_name);

CREATE OR REPLACE VIEW g_fireboxinfotable (
  ip_address, appliance_name, type, model_name, sn, timezone, version,
  active, cluster, cstate, when_join, update_time)
AS SELECT
  ip_address, appliance_name, type, model_name, sn, timezone, version,
  active, cluster, text '', when_join, update_time
FROM g_deviceinfo;

CREATE TABLE group_name(
   id               INTEGER DEFAULT NEXTVAL('appliance_id_seq') PRIMARY KEY,
   group_name       TEXT NOT NULL,
   description      TEXT
) WITH (FILLFACTOR = 50);
CREATE UNIQUE INDEX group_name_gn_idx ON group_name(group_name);

CREATE TABLE group_appliance(
   group_id         INTEGER REFERENCES group_name(id) ON DELETE CASCADE,
   appliance_id     INTEGER
) WITH (FILLFACTOR = 50);
CREATE INDEX group_appliance_gid_idx ON group_appliance(group_id);
CREATE INDEX group_appliance_aid_idx ON group_appliance(appliance_id);

CREATE TABLE g_versiontable (
   company          TEXT,
   major_number     INTEGER NOT NULL,
   minor_number     INTEGER NOT NULL,
   copyright        TEXT,
   install_time     TIMESTAMP DEFAULT (CURRENT_TIMESTAMP AT TIME ZONE 'UTC')
);

CREATE TABLE g_base_config(
    maximum_storage     INTEGER NOT NULL DEFAULT 100,
    delete_data_by_age  INTEGER NOT NULL DEFAULT -1,
    public_ips          TEXT NOT NULL DEFAULT '',
    public_port         INTEGER NOT NULL DEFAULT 443,
    last_backup_runtime TIMESTAMP,
    run_backup_day      INTEGER DEFAULT 1,
    run_backup_time     TIME DEFAULT '02:30:00'::TIME
);
INSERT INTO g_base_config (maximum_storage, delete_data_by_age, public_ips,
    public_port, last_backup_runtime, run_backup_day, run_backup_time)
SELECT 100, -1, '', 443, NULL, 1, '02:30:00'::TIME WHERE
NOT EXISTS (SELECT * FROM g_base_config LIMIT 1);

CREATE TABLE g_ConfigureTable(
  maximum_storage          INTEGER DEFAULT 100,
  warning_limit            INTEGER DEFAULT 80,
  days_retain_log          INTEGER DEFAULT 30,
  days_backup_log          INTEGER DEFAULT 1,
  days_retain_audit        INTEGER DEFAULT 365,
  backup_file_dir          TEXT,   
  backup_file_format       TEXT DEFAULT 'CSV',   
  run_backup_time          TIME DEFAULT '02:30',
  run_backup_day           INTEGER DEFAULT 1,
  latest_backup_runtime    TIMESTAMP,
  run_delete_time          TIME DEFAULT '02:30',
  run_delete_day           INTEGER DEFAULT 1,
  latest_delete_runtime    TIMESTAMP,
  logserver_pause          INTEGER DEFAULT 0,
  notify_admin_flag        INTEGER DEFAULT 0,
  smtp_login_flag          INTEGER DEFAULT 0,
  log_del_flag             INTEGER DEFAULT 0,
  show_schedule_flag       INTEGER DEFAULT 0,
  backup_flag              INTEGER DEFAULT 0,
  show_backup_flag         INTEGER DEFAULT 0,
  appliance_notify_flag    INTEGER DEFAULT 0,
  test_email_flag          INTEGER DEFAULT 0,
  delete_data_in_table     INTEGER DEFAULT 0,
  notify_on_failure_flag   INTEGER DEFAULT 0,
  dynamic_host_lookup_flag INTEGER DEFAULT 0,
  delete_log_by_age        INTEGER DEFAULT -1
);

INSERT INTO g_ConfigureTable
   SELECT 100, 80, 30, 1, 365, 'c:/Documents and Settings/WatchGuard/wlogserver/tmp',
           'CSV','02:30', 1, NULL, '02:30', 1, NULL, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -1
   WHERE NOT EXISTS (SELECT * FROM g_ConfigureTable LIMIT 1);

CREATE TABLE g_EmailTable(
  email_address     TEXT,
  email_host        TEXT,
  user_id           TEXT,
  password          TEXT,
  email_subject     TEXT,
  email_body        TEXT,
  from_address      TEXT,
  to_address        TEXT,
  update_time       TIMESTAMP DEFAULT (CURRENT_TIMESTAMP AT TIME ZONE 'UTC'),
  starttls_flag     INTEGER DEFAULT 0
);

INSERT INTO g_EmailTable
    SELECT '', 'localhost', NULL, NULL, 'WatchGuard Log Server Notification', 'WatchGuard Notification',
           'wlogserver@localhost', 'admin@localhost'
    WHERE NOT EXISTS (SELECT * FROM g_EmailTable LIMIT 1);

CREATE TABLE g_cluster_roles(
   role_id   INTEGER NOT NULL PRIMARY KEY,
   role_name TEXT NOT NULL);
INSERT INTO g_cluster_roles (role_id, role_name) SELECT 0, 'standalone' WHERE
   NOT EXISTS (SELECT 1 FROM g_cluster_roles WHERE role_id = 0);
INSERT INTO g_cluster_roles (role_id, role_name) SELECT 1, 'worker' WHERE
   NOT EXISTS (SELECT 1 FROM g_cluster_roles WHERE role_id = 1);
INSERT INTO g_cluster_roles (role_id, role_name) SELECT 2, 'backup master' WHERE
   NOT EXISTS (SELECT 1 FROM g_cluster_roles WHERE role_id = 2);
INSERT INTO g_cluster_roles (role_id, role_name) SELECT 3, 'master' WHERE
   NOT EXISTS (SELECT 1 FROM g_cluster_roles WHERE role_id = 3);
INSERT INTO g_cluster_roles (role_id, role_name) SELECT 4, 'idle' WHERE
   NOT EXISTS (SELECT 1 FROM g_cluster_roles WHERE role_id = 4);
INSERT INTO g_cluster_roles (role_id, role_name) SELECT 5, 'standby' WHERE
   NOT EXISTS (SELECT 1 FROM g_cluster_roles WHERE role_id = 5);

CREATE TABLE g_modelinfo(
   model_name_short        TEXT,
   model_name              TEXT
);
-- g_modelinfo is populated after this file is loaded

CREATE TABLE g_registered_instances(
    id          TEXT NOT NULL,
    component   TEXT NOT NULL,
    ip          TEXT DEFAULT host(inet_client_addr()),
    update_time TIMESTAMP NOT NULL DEFAULT date_trunc('second', timezone('utc', clock_timestamp()))
);
ALTER TABLE g_registered_instances ADD PRIMARY KEY (id, component);

CREATE sequence g_connected_clients_id_seq start with 1;
CREATE TABLE g_connected_clients(
    id           BIGINT NOT NULL DEFAULT nextval('g_connected_clients_id_seq') PRIMARY KEY,
    appliance_id INTEGER NOT NULL,
    instance_id  TEXT NOT NULL,
    component    TEXT NOT NULL,
    update_time  TIMESTAMP NOT NULL DEFAULT date_trunc('second', timezone('utc', clock_timestamp()))
);
ALTER TABLE g_connected_clients ADD CONSTRAINT g_connected_clients_fk
    FOREIGN KEY(instance_id, component) REFERENCES g_registered_instances(id, component);
CREATE INDEX g_connected_clients_aid_idx ON g_connected_clients(appliance_id);
CREATE INDEX g_connected_clients_component_idx ON g_connected_clients(component);
CREATE INDEX g_connected_clients_iid_idx ON g_connected_clients(instance_id);
CREATE INDEX g_connected_clients_ut_idx ON g_connected_clients(update_time);

CREATE TABLE audit_events(
    id          TEXT PRIMARY KEY,
    description TEXT NOT NULL
);
-- audit_events is populated after this file is loaded

CREATE SEQUENCE audit_log_id_seq START WITH 1;
CREATE TABLE audit_log(
    id               BIGINT PRIMARY KEY DEFAULT nextval('audit_log_id_seq'),
    update_time      TIMESTAMP NOT NULL DEFAULT date_trunc('second', timezone('utc', clock_timestamp())),
    user_name        TEXT,
    user_comment     TEXT,
    ip_address       TEXT,
    object_type      TEXT,
    object_id        TEXT,
    event_id         TEXT NOT NULL,
    msg              TEXT NOT NULL);
CREATE INDEX audit_log_ut_idx ON audit_log(update_time);
CREATE INDEX audit_log_user_name_idx ON audit_log(user_name);
CREATE INDEX audit_log_object_type_id_idx ON audit_log(object_type, object_id);
CREATE INDEX audit_log_event_id_idx ON audit_log(event_id);

CREATE TABLE g_feature_key_status ( 
    appliance_id  INTEGER NOT NULL PRIMARY KEY,
    feature_key   TEXT,
    update_time   TIMESTAMP WITHOUT TIME ZONE NOT NULL);
